/*:ja
 * @plugindesc 数値ピクチャ表示プラグイン
 * @author TRS
 * *
 * @param ピクチャ番号
 * @desc 数値のピクチャ番号の開始位置
 * 指定先から3つ番号（3桁分）を取得します
 * @type number
 * @max 100
 * @min 1
 * @default 1
 *
 * @param 読み込む番号ピクチャの基本名称
 * @desc 基本名称の後ろに"_(番号)"をつけて読み込みます
 * @type string
 * @default num
 *
 * @param ピクチャサイズ
 * @desc 番号ピクチャのサイズ
 * @type number
 * @max 96
 * @min 12
 * @default 48
 *
 * @param 次の数字までの間隔
 * @desc ピクチャサイズを基準にどれだけずらすか？
 * @type number
 * @max 20
 * @min 1
 * @default 1
 *
 * @param 番号ピクチャのX座標
 * @desc 左上原点で、数値ピクチャ右端の横座標
 * @type number
 * @max 816
 * @min 0 
 * @default 100
 *
 * @param 番号ピクチャのY座標
 * @desc 左上原点で、数値ピクチャ上の縦座標
 * @type number
 * @max 624
 * @min 0 
 * @default 50
 *
 * @param 数値変数
 * @desc ピクチャ化する数値の入っているゲーム変数
 * @type variable
 * @default 1 
 * @default 50
 *
 * @help
 *
 * 変数内に保存されている数値を数値ピクチャで表示します
 *
 * -----------------------------
 * プラグインコマンド
 * 
 *  数値ピクチャ表示
 *  ・数値変数の中身をピクチャとして出します
 *
 *  数値ピクチャ消去
 *  ・数値ピクチャを完全消去します
 *
 * ------------------------------
 */
 
(function()
{
	var parameters = PluginManager.parameters('PictureNumber');
	var param = {};
	
	//基本設定
	param.pictureNo = Number( parameters['ピクチャ番号'] || 1 );
	param.commonName = String( parameters['読み込む番号ピクチャの基本名称'] );
	param.fontPicSize = Number( parameters['ピクチャサイズ'] || 48 );
	param.offset = Number( parameters['次の数字までの間隔'] || 48 );
	param.Xpos = Number( parameters['番号ピクチャのX座標'] || 100 );
	param.Ypos = Number( parameters['番号ピクチャのY座標'] || 50 );
	param.valNum = Number( parameters['数値変数'] || 1 );
	
	//-----------------------------------------------------------------------------
    //	 初期設定
    //
    var _game_interpreter_prototype_initialize = Game_Interpreter.prototype.initialize;
    
	Game_Interpreter.prototype.initialize = function(depth) 
	{
	   _game_interpreter_prototype_initialize.call(this);
	   this.InitLoadChkFlg = false;
	};

	//-----------------------------------------------------------------------------
    //	 Game_Interpreter (プラグインコマンド用インタプリタ)
    //
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args)
	{
		_Game_Interpreter_pluginCommand.call(this, command, args);
		
		//ゲージ更新
		if( command === "数値ピクチャ表示")
		{
			this.InitLoadChk();					//最初に読み込めるか確認
			this.numPicUpdate();
		}
		else if( command === "数値ピクチャ消去" )
		{
			this.delNumPic();
		}
	}
	
	/**
	 * ピクチャファイル存在チェックと先行ロード
	 */
	Game_Interpreter.prototype.InitLoadChk = function()
	{
		if( this.InitLoadChkFlg === false)
		{
			this.NumFileTable = [];
			
			for(var i = 0; i < 10; i++)
			{
				$gameScreen.showPicture( param.pictureNo, param.commonName + "_" + i, 0, 0, 0, 0, 0, 0, 0 );
				this.NumFileTable[i] = param.commonName + "_" + i;					//ファイル名インデックスにしまう
				
			}
			
			this.InitLoadChkFlg = true;
		}
	}
	
	/**
	 * 番号ピクチャ出すように編集
	 */
	Game_Interpreter.prototype.numPicUpdate = function()
	{
		var varStrValNum = String( $gameVariables.value(param.valNum) );
		var varLen = varStrValNum.length;
		
		var lstNumber = [];
		
		//ここで配列に格納
		for(var i = 0; i < varLen; i++)
		{
			lstNumber[i] = varStrValNum.substr(i, 1);
		}
		
		this.drawPicNumber( lstNumber );
		
	}
	
	/**
	 * ピクチャ表示
	 */
	Game_Interpreter.prototype.drawPicNumber = function( lstNumber )
	{
		var varPicNo;
		var varFileName;
		var varSumXpos;
		
		for(var i= 0; i < lstNumber.length; i++)
		{
			varPicNo = param.pictureNo + i;								//ピクチャ番号
			varFileName = this.NumFileTable[ Number( lstNumber[i] ) ];	//取得してる数値を個々に番号にする
			varSumXpos = param.Xpos - (param.fontPicSize + param.offset) * lstNumber.length + ((param.fontPicSize + param.offset) * i);
			
			$gameScreen.showPicture( varPicNo, varFileName, 0, varSumXpos, param.Ypos, 100, 100, 255, 0 );
		}
	}
	
	/**
	 * 消す処理
	 */
	Game_Interpreter.prototype.delNumPic = function()
	{
		$gameScreen.erasePicture(param.pictureNo);
		$gameScreen.erasePicture(param.pictureNo + 1);
		$gameScreen.erasePicture(param.pictureNo + 2);
		
		this.InitLoadChkFlg = false;

	}
	
})();
 